using System;
using Roblox.Platform.Assets;

namespace Roblox.Thumbs;

public interface IAssetThumbnail
{
	ThumbnailDomainFactories DomainFactories { get; }

	bool IsSupported3DAssetType(int assetTypeId);

	bool IsSupportedAnimatedAssetType(int assetTypeId);

	/// <summary>
	/// Returns Thumb Result for 3D thumbnail request for moderation purposes (moderation agnostic)
	/// </summary>
	/// <param name="assetHashId">asset hash id</param>
	/// <returns>A <see cref="T:Roblox.Thumbs.ThumbResult" /> for specified assetHashId</returns>
	ThumbResult GetModerationAgnostic3DThumbnailUrlForAssetHash(long assetHashId);

	/// <summary>
	/// Returns Thumb Result for 2D thumbnail request for moderation purposes (moderation agnostic)
	/// </summary>
	/// <param name="assetHashId">asset hash id</param>
	/// <param name="imageParameters">A <see cref="T:Roblox.Thumbs.ImageParameters" /> (contains format, width, height, etc.)</param>
	/// <returns>A <see cref="T:Roblox.Thumbs.ThumbResult" /> for specified assetHashId and imageParameters</returns>
	ThumbResult GetModerationAgnosticThumbnailUrlForAssetHash(long assetHashId, ImageParameters imageParameters);

	ThumbResult GetAsset3DThumbnailUrl(long assetId, string imageFormat = "Obj");

	ThumbResult GetAssetAnimatedThumbnailUrl(long assetId);

	ThumbResult GetThumbnailUrl(IAsset asset, ImageParameters imageParameters);

	ThumbResult GetGameIconThumbnailUrl(IAsset asset, ImageParameters imageParameters);

	/// <summary>
	/// Returns a substitute <see cref="T:Roblox.Thumbs.ThumbResult" /> if the auto-generated thumbnail should not be used for some reason.
	/// For example: If the thumbnail has been moderated, if the thumbnail is awaiting moderation, if the asset is a place
	/// and we now default to canned images, if the user has uploaded a replacement icon, etc.
	/// </summary>
	/// <param name="asset">The asset.</param>
	/// <param name="thumbResult">If this thumbnail should be substituted, the substitution is returned here.</param>
	/// <param name="substituteAsset"></param>
	/// <param name="imageParameters">The image parameters.</param>
	/// <param name="overrideModeration">Whether to override moderation.</param>
	/// <param name="ignoreAssetMediaItems">Whether to ignore place media items. If set to True, use default thumbnail.</param>
	/// <param name="returnAutoGenerated">Whether to return auto generated thumbnail when there is no media items. When set to true, return camera generated thumbnail. Only applied to place.</param>
	/// <param name="returnGameIcon">Whether to return a default game icon when there is no media items and returnAutoGenerated is not set to true.</param>
	/// <returns></returns>
	bool TryThumbnailSubstitution(IAsset asset, out ThumbResult thumbResult, ref IAsset substituteAsset, ImageParameters imageParameters, bool overrideModeration, bool ignoreAssetMediaItems = false, bool returnAutoGenerated = false, bool returnGameIcon = false);

	/// <summary>
	/// Get multiple thumbnails at one time.
	/// </summary>
	/// <param name="assets">The assets who's thumbnails to get.</param>
	/// <param name="imageParameters">The image parameters.</param>
	/// <param name="overrideModeration">Whether to override moderation.</param>
	/// <param name="ignoreAssetMediaItems">Whether to ignore place media items. If set to True, use default thumbnail.</param>
	/// <param name="returnAutoGenerated">Whether to return auto generated thumbnail when there is no media items. When set to true, return camera generated thumbnail. Only applied to place.</param>
	/// <param name="returnGameIcon">Whether to return a default game icon when there is no media items and returnAutoGenerated is not set to true.</param>
	/// <returns>An array of <see cref="T:Roblox.Thumbs.ThumbResult" />.</returns>
	/// <exception cref="T:System.ArgumentNullException">imageParameters</exception>
	/// <exception cref="T:System.ArgumentException">assets</exception>
	[Obsolete("Please use the MultigetThumbnailUrl that accepts Platform.Assets.IAssets instead")]
	ThumbResult[] MultiGetThumbnailUrl(IAsset[] assets, ImageParameters imageParameters, bool overrideModeration, bool ignoreAssetMediaItems = false, bool returnAutoGenerated = false, bool returnGameIcon = false);

	/// <summary>
	/// Get multiple thumbnails at one time. The returned array will preserve input order, but will remove duplicate thumbnail URLs, if multiple Assets map to the same URL.
	/// </summary>
	/// <param name="assets">The The Platform <see cref="T:Roblox.Platform.Assets.IAsset" /> assets who's thumbnails to get.</param>
	/// <param name="imageParameters">The image parameters.</param>
	/// <param name="overrideModeration">Whether to override moderation.</param>
	/// <param name="ignoreAssetMediaItems">Whether to ignore place media items. If set to True, use default thumbnail.</param>
	/// <param name="returnAutoGenerated">Whether to return auto generated thumbnail when there is no media items. When set to true, return camera generated thumbnail. Only applied to place.</param>
	/// <param name="returnGameIcon">Whether to return a default game icon when there is no media items and returnAutoGenerated is not set to true.</param>
	/// <returns>An array of <see cref="T:Roblox.Thumbs.ThumbResult" />.</returns>
	/// <exception cref="T:System.ArgumentNullException"><paramref name="imageParameters" /></exception>
	/// <exception cref="T:System.ArgumentNullException"><paramref name="assets" /></exception>
	ThumbResult[] MultiGetThumbnailUrl(Roblox.Platform.Assets.IAsset[] assets, ImageParameters imageParameters, bool overrideModeration, bool ignoreAssetMediaItems = false, bool returnAutoGenerated = false, bool returnGameIcon = false);

	/// <summary>
	/// Get thumbnail url for the given asset.
	/// </summary>
	/// <param name="asset">The asset.</param>
	/// <param name="imageParameters">The image parameters.</param>
	/// <param name="overrideModeration">Whether to override moderation.</param>
	/// <param name="ignoreAssetMediaItems">Whether to ignore place media items. If set to True, use default thumbnail.</param>
	/// <param name="returnAutoGenerated">Whether to return auto generated thumbnail when there is no media items. When set to true, return camera generated thumbnail. Only applied to place.</param>
	/// <param name="returnGameIcon">Whether to return a default game icon when there is no media items and returnAutoGenerated is not set to true.</param>
	/// <returns></returns>
	ThumbResult GetThumbnailUrl(IAsset asset, ImageParameters imageParameters, bool overrideModeration, bool ignoreAssetMediaItems = false, bool returnAutoGenerated = false, bool returnGameIcon = false);

	/// <summary>
	/// Get thumbnail url for the given asset.
	/// </summary>
	/// <param name="asset">The Platform <see cref="T:Roblox.Platform.Assets.IAsset" />.</param>
	/// <param name="imageParameters">The image parameters.</param>
	/// <param name="overrideModeration">Whether to override moderation.</param>
	/// <param name="ignoreAssetMediaItems">Whether to ignore place media items. If set to True, use default thumbnail.</param>
	/// <param name="returnAutoGenerated">Whether to return auto generated thumbnail when there is no media items. When set to true, return camera generated thumbnail. Only applied to place.</param>
	/// <param name="returnGameIcon">Whether to return a default game icon when there is no media items and returnAutoGenerated is not set to true.</param>
	/// <returns><see cref="T:Roblox.Thumbs.ThumbResult" /></returns>
	ThumbResult GetThumbnailUrl(Roblox.Platform.Assets.IAsset asset, ImageParameters imageParameters, bool overrideModeration, bool ignoreAssetMediaItems = false, bool returnAutoGenerated = false, bool returnGameIcon = false);

	string GetAssetThumbnailRetryUrl(bool final, long? assetId, int thumbWidth, int thumbHeight, string thumbFormat);

	/// <summary>
	/// Used for displaying Thumbnails that have been explicitly set as part of Curated Games Snapshot
	/// </summary>
	/// <param name="assetHashId"></param>
	/// <param name="imageParameters"></param>
	/// <returns></returns>
	ThumbResult GetThumbnailUrlByAssetHashIdForSnapshots(long assetHashId, ImageParameters imageParameters);

	ThumbResult GetPlaceThumbIgnoreAssetMedia(IAsset placeAsset, ImageParameters imageParameters, bool overrideModeration);

	ThumbResult GetDefaultPlaceThumbnail(long placeId, ImageParameters imageParameters);

	ThumbResult GetDefaultPlaceGameIcon(long placeId, ImageParameters imageParameters);
}
