using System;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Drawing.Imaging;
using System.IO;
using Roblox.ApiClientBase;
using Roblox.AssetMedia;
using Roblox.Drawing;
using Roblox.EphemeralCounters;
using Roblox.FloodCheckers;
using Roblox.FloodCheckers.Core;
using Roblox.Platform.AssetMedia.Properties;
using Roblox.Platform.AssetOwnership;
using Roblox.Platform.Assets;
using Roblox.Platform.Assets.Interface;
using Roblox.Platform.Core;
using Roblox.Platform.Membership;
using Roblox.Thumbs;

namespace Roblox.Platform.AssetMedia;

/// <summary>
/// Implements <see cref="T:Roblox.Platform.AssetMedia.IPlaceThumbnailUploader" />.
/// </summary>
public class PlaceThumbnailUploader : IPlaceThumbnailUploader
{
	private readonly IAssetThumbnail _AssetThumbnail;

	private readonly IAssetOwnershipAuthority _AssetOwnershipAuthority;

	private readonly IEphemeralCounterFactory _EphemeralCounterFactory;

	private readonly IImageFactory _ImageFactory;

	private readonly IUploadFloodcheckerFactory _UploadFloodcheckerFactory;

	private const string _AddGeneratedImageAttemptCounterName = "AddGeneratedImage_Attempt";

	private const string _AddGeneratedImageProcessingTimeCounterName = "AddGeneratedImage_ProccesingTime";

	private static readonly ImageParameters _ThumbnailParameters = new ImageParameters(768, 432, ImageFormat.Png);

	private static TimeSpan DownloadTimeout => Settings.Default.DownloadImageTimeout;

	[ExcludeFromCodeCoverage]
	internal virtual int MaximumUploadedImageWidthOrHeight => Settings.Default.MaximumUploadedImageWidthOrHeight;

	/// <summary>
	/// Constructs the place thumbnail uploader.
	/// </summary>
	/// <param name="assetThumbnail">An <see cref="T:Roblox.Thumbs.IAssetThumbnail" />.</param>
	/// <param name="assetOwnershipAuthority">An <see cref="T:Roblox.Platform.AssetOwnership.IAssetOwnershipAuthority" />.</param>
	/// <param name="ephemeralCounterFactory">An <see cref="T:Roblox.EphemeralCounters.IEphemeralCounterFactory" />.</param>
	/// <param name="imageFactory">An <see cref="T:Roblox.Platform.Assets.IImageFactory" />.</param>
	/// <param name="uploadFloodcheckerFactory">An <see cref="T:Roblox.Platform.Assets.Interface.IUploadFloodcheckerFactory" /></param>
	/// <exception cref="T:System.ArgumentNullException">
	/// - <paramref name="assetThumbnail" />
	/// - <paramref name="assetOwnershipAuthority" />
	/// - <paramref name="ephemeralCounterFactory" />
	/// - <paramref name="imageFactory" />
	/// - <paramref name="uploadFloodcheckerFactory" />
	/// </exception>
	public PlaceThumbnailUploader(IAssetThumbnail assetThumbnail, IAssetOwnershipAuthority assetOwnershipAuthority, IEphemeralCounterFactory ephemeralCounterFactory, IImageFactory imageFactory, IUploadFloodcheckerFactory uploadFloodcheckerFactory)
	{
		_AssetThumbnail = assetThumbnail ?? throw new ArgumentNullException("assetThumbnail");
		_AssetOwnershipAuthority = assetOwnershipAuthority ?? throw new ArgumentNullException("assetOwnershipAuthority");
		_EphemeralCounterFactory = ephemeralCounterFactory ?? throw new ArgumentNullException("ephemeralCounterFactory");
		_ImageFactory = imageFactory ?? throw new ArgumentNullException("imageFactory");
		_UploadFloodcheckerFactory = uploadFloodcheckerFactory ?? throw new ArgumentNullException("uploadFloodcheckerFactory");
	}

	/// <inheritdoc cref="M:Roblox.Platform.AssetMedia.IPlaceThumbnailUploader.AddGeneratedPlaceThumbnail(Roblox.Platform.Assets.IPlace,Roblox.Platform.Membership.IUser)" />
	public IPlaceThumbnail AddGeneratedPlaceThumbnail(IPlace place, IUser actingUser)
	{
		if (place == null)
		{
			throw new ArgumentNullException("place");
		}
		if (actingUser == null)
		{
			throw new ArgumentNullException("actingUser");
		}
		ThumbResult autoThumbnailResult = _AssetThumbnail.GetThumbnailUrl(place, _ThumbnailParameters, overrideModeration: false, ignoreAssetMediaItems: true, returnAutoGenerated: true);
		if (autoThumbnailResult == null || !autoThumbnailResult.final)
		{
			throw new PlatformOperationUnavailableException("Generated place thumbnail is not available");
		}
		IFloodChecker generatedIconUploadFloodChecker = CreatePlaceThumbnailUploadFloodChecker(place);
		if (generatedIconUploadFloodChecker.IsFlooded())
		{
			throw new PlatformFloodedException("Too many place thumbnail upload requests");
		}
		IncrementEphemeralCounter("AddGeneratedImage_Attempt");
		Stopwatch stopWatch = Stopwatch.StartNew();
		string downloadUrl = autoThumbnailResult.GetUrl(arg: true);
		IPlaceThumbnail placeThumbnail;
		using (MemoryStream stream = new MemoryStream(DownloadImageData(downloadUrl)))
		{
			placeThumbnail = AddThumbnailFromStream(place, actingUser, stream);
		}
		generatedIconUploadFloodChecker.UpdateCount();
		stopWatch.Stop();
		AddProcessingTimeToEphemeralCounter("AddGeneratedImage_ProccesingTime", stopWatch.Elapsed.TotalMilliseconds);
		return placeThumbnail;
	}

	/// <inheritdoc cref="M:Roblox.Platform.AssetMedia.IPlaceThumbnailUploader.AddGeneratedPlaceThumbnail(Roblox.Platform.Assets.IPlace,Roblox.Platform.Membership.IUser)" />
	public IPlaceThumbnail AddPlaceThumbnail(IPlace place, IUser actingUser, Stream imageData)
	{
		if (place == null)
		{
			throw new ArgumentNullException("place");
		}
		if (imageData == null)
		{
			throw new ArgumentNullException("imageData");
		}
		if (actingUser == null)
		{
			throw new ArgumentNullException("actingUser");
		}
		IFloodChecker imageUploadFloodChecker = _UploadFloodcheckerFactory.GetFreeThumbnailImageUploadFloodChecker(actingUser);
		if (imageUploadFloodChecker.IsFlooded())
		{
			throw new PlatformFloodedException("Too many place icon upload requests");
		}
		IPlaceThumbnail result = AddThumbnailFromStream(place, actingUser, imageData);
		imageUploadFloodChecker.UpdateCount();
		return result;
	}

	[ExcludeFromCodeCoverage]
	internal virtual IFloodChecker CreatePlaceThumbnailUploadFloodChecker(IPlace place)
	{
		return new GeneratedThumbnailUploadFloodChecker(place.Id);
	}

	[ExcludeFromCodeCoverage]
	internal virtual byte[] DownloadImageData(string downloadUrl)
	{
		using TimeoutWebClient webClient = new TimeoutWebClient(DownloadTimeout);
		return webClient.DownloadData(downloadUrl);
	}

	[ExcludeFromCodeCoverage]
	internal virtual long AddPlaceMedia(IPlace place, IImage image, IUser actingUser)
	{
		return PlaceMediaManager.AddPlaceMedia(place.Id, image.Id, actingUser.Id).ID;
	}

	private IPlaceThumbnail AddThumbnailFromStream(IPlace place, IUser actingUser, Stream stream)
	{
		AssetCreatorInfo assetCreatorInfo = new AssetCreatorInfo(Roblox.Platform.Assets.CreatorType.User, actingUser.Id);
		TrustedAssetTextInfo assetNameAndDescription = new TrustedAssetTextInfo($"{place.Name}_Image", string.Empty);
		Stream resampledTexture;
		try
		{
			resampledTexture = ImageUtil.ResampleTextureEnforceMaxEdgeSize(stream, MaximumUploadedImageWidthOrHeight);
		}
		catch (ArgumentException)
		{
			throw new InvalidDataException("The provided image data stream is in an invalid format");
		}
		IImage image;
		using (resampledTexture)
		{
			StreamCreatorInfo streamCreatorInfo = new StreamCreatorInfo(resampledTexture);
			image = _ImageFactory.CreateWithTrustedAssetText(assetNameAndDescription, assetCreatorInfo, streamCreatorInfo, actingUser);
		}
		_AssetOwnershipAuthority.AwardAsset(image.Id, actingUser.Id);
		long mediaItemId = AddPlaceMedia(place, image, actingUser);
		return new PlaceThumbnail
		{
			Id = mediaItemId,
			PlaceId = place.Id,
			ImageId = image.Id
		};
	}

	private void IncrementEphemeralCounter(string counterName)
	{
		_EphemeralCounterFactory.GetCounter(counterName).IncrementInBackground(1, (Action<Exception>)null);
	}

	private void AddProcessingTimeToEphemeralCounter(string counterName, double timeInMilliseconds)
	{
		ISequence counter = _EphemeralCounterFactory.GetSequence(counterName);
		RobloxThreadPool.QueueUserWorkItem(delegate
		{
			counter.Add(timeInMilliseconds);
		});
	}
}
