using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading;
using Roblox.ApiClientBase;
using Roblox.AssetMedia;
using Roblox.Drawing;
using Roblox.EphemeralCounters;
using Roblox.FloodCheckers.Core;
using Roblox.Platform.Assets;
using Roblox.Platform.Assets.Interface;
using Roblox.Platform.Core;
using Roblox.Platform.Membership;
using Roblox.Thumbs;

namespace Roblox.Platform.AssetMedia;

internal class AssetMediaThumbnailUploader : IAssetMediaThumbnailUploader
{
	private const string AddGeneratedModelProccesingTime = "AddGeneratedModel_ProccesingTime";

	private const string AddGeneratedModelAttempt = "AddGeneratedModel_Attempt";

	private const string AddGeneratedModelError = "AddGeneratedModel_Error";

	internal readonly AssetMediaDomainFactories DomainFactories;

	internal readonly ThumbnailDomainFactories ThumbnailDomainFactories;

	private readonly IUploadFloodcheckerFactory _UploadFloodCheckerFactory;

	public AssetMediaThumbnailUploader(AssetMediaDomainFactories domainFactories, ThumbnailDomainFactories thumbnailDomainFactories, IUploadFloodcheckerFactory uploadFloodCheckerFactory)
	{
		DomainFactories = domainFactories ?? throw new ArgumentNullException("domainFactories");
		ThumbnailDomainFactories = thumbnailDomainFactories ?? throw new ArgumentNullException("thumbnailDomainFactories");
		_UploadFloodCheckerFactory = uploadFloodCheckerFactory ?? throw new ArgumentNullException("uploadFloodCheckerFactory");
	}

	/// <inheritdoc cref="M:Roblox.Platform.AssetMedia.IAssetMediaThumbnailUploader.GenerateModelImage(Roblox.Asset)" />
	public ThumbResult GenerateModelImage(Asset asset)
	{
		ImageParameters imageParameters = new ImageParameters(DomainFactories.Settings.GeneratedThumbnailWidth, DomainFactories.Settings.GeneratedThumbnailHeight, "Png");
		return ThumbnailDomainFactories.Asset.GetThumbnailUrl(asset, imageParameters, overrideModeration: true, ignoreAssetMediaItems: true, returnAutoGenerated: true);
	}

	/// <inheritdoc cref="M:Roblox.Platform.AssetMedia.IAssetMediaThumbnailUploader.UploadGeneratedAssetMediaImageAsync(Roblox.Asset,Roblox.Platform.Membership.IUser)" />
	public void UploadGeneratedAssetMediaImageAsync(Asset asset, IUser authenticatedUser)
	{
		RobloxThreadPool.QueueUserWorkItem(delegate
		{
			UploadGeneratedAssetMediaImageAndUploadCounters(asset, authenticatedUser);
		});
	}

	/// <inheritdoc cref="M:Roblox.Platform.AssetMedia.IAssetMediaThumbnailUploader.UploadAssetMediaImage(System.IO.Stream,Roblox.Asset,Roblox.Platform.Membership.IUser,System.Boolean)" />
	public IImage UploadAssetMediaImage(Stream imageStream, Asset asset, IUser user, bool isPlace)
	{
		IImage image = UploadAssetMediaImage(imageStream, asset.Name, user);
		if (isPlace)
		{
			PlaceMediaManager.AddPlaceMedia(asset.ID, image.Id, user.Id);
		}
		else
		{
			AssetMediaManager.AddAssetMedia(asset.ID, image.Id, user.Id);
		}
		return image;
	}

	/// <inheritdoc cref="M:Roblox.Platform.AssetMedia.IAssetMediaThumbnailUploader.UploadAssetMediaImage(System.IO.Stream,Roblox.Platform.Assets.IAsset,Roblox.Platform.Membership.IUser)" />
	public IImage UploadAssetMediaImage(Stream imageStream, Roblox.Platform.Assets.IAsset asset, IUser user)
	{
		if (imageStream == null)
		{
			throw new ArgumentNullException("imageStream");
		}
		if (asset == null)
		{
			throw new ArgumentNullException("asset");
		}
		if (user == null)
		{
			throw new ArgumentNullException("user");
		}
		IRetryAfterFloodChecker floodChecker = _UploadFloodCheckerFactory.GetImageUploadFloodchecker(user);
		if (floodChecker.IsFlooded())
		{
			throw new PlatformFloodedException();
		}
		IImage result = UploadAssetMediaImage(imageStream, asset.Name, user);
		floodChecker.UpdateCount();
		return result;
	}

	/// <inheritdoc cref="M:Roblox.Platform.AssetMedia.IAssetMediaThumbnailUploader.AddAssetMediaImage(Roblox.Platform.Assets.IAsset,Roblox.Platform.Assets.IImage,Roblox.Platform.Membership.IUser)" />
	public void AddAssetMediaImage(Roblox.Platform.Assets.IAsset asset, IImage image, IUser user)
	{
		Roblox.Platform.Assets.AssetType? assetType = asset.GetAssetType(DomainFactories.AssetDomainFactories.AssetTypeFactory);
		if (assetType == Roblox.Platform.Assets.AssetType.Place)
		{
			PlaceMediaManager.AddPlaceMedia(asset.Id, image.Id, user.Id);
		}
		else
		{
			AssetMediaManager.AddAssetMedia(asset.Id, image.Id, user.Id, assetType);
		}
	}

	private IImage UploadAssetMediaImage(Stream imageStream, string assetName, IUser user)
	{
		IImage image;
		using (Stream resampledTexture = ImageUtil.ResampleTextureEnforceMaxEdgeSize(imageStream, DomainFactories.Settings.MaximumUploadedImageWidthOrHeight))
		{
			TrustedAssetTextInfo assetTextInfo = new TrustedAssetTextInfo(assetName + "_Image", string.Empty);
			AssetCreatorInfo assetCreatorInfo = new AssetCreatorInfo(Roblox.Platform.Assets.CreatorType.User, user.Id);
			StreamCreatorInfo streamCreatorInfo = new StreamCreatorInfo(resampledTexture);
			image = DomainFactories.AssetDomainFactories.ImageFactory.CreateWithTrustedAssetText(assetTextInfo, assetCreatorInfo, streamCreatorInfo, user);
		}
		DomainFactories.AssetOwnershipAuthority.AwardAsset(image.Id, user.Id);
		return image;
	}

	private TimeSpan DoUpload(Asset asset, IUser authenticatedUser)
	{
		Stopwatch stopWatch = new Stopwatch();
		stopWatch.Start();
		ThumbResult thumbResult = GenerateModelImage(asset);
		while (!thumbResult.final)
		{
			if (stopWatch.Elapsed > DomainFactories.Settings.DefaultCameraGeneratedThumbnailTimeout)
			{
				throw new ApplicationException("Error: thumbnail generation took longer than the allowed limit.");
			}
			thumbResult = GenerateModelImage(asset);
			Thread.Sleep(1000);
		}
		stopWatch.Stop();
		TimeSpan duration = stopWatch.Elapsed;
		byte[] imageData;
		using (TimeoutWebClient wc = new TimeoutWebClient(DomainFactories.Settings.DownloadImageTimeout))
		{
			imageData = wc.DownloadData(thumbResult.GetUrl(arg: true));
		}
		using (MemoryStream stream = new MemoryStream(imageData))
		{
			using Stream resampledTexture = ImageUtil.ResampleTextureEnforceDesiredSizeWithPadding(stream, DomainFactories.Settings.GeneratedThumbnailWidth, DomainFactories.Settings.GeneratedThumbnailHeight);
			AssetMediaItem mediaItem = AssetMediaManager.GetAssetMediaItemsByAssetID(asset.ID).FirstOrDefault((AssetMediaItem m) => Asset.Get(m.MediaAssetID).AssetTypeID == AssetType.ImageID);
			if (mediaItem != null)
			{
				StreamCreatorInfo streamCreatorInfo = new StreamCreatorInfo(resampledTexture);
				IImage image = DomainFactories.AssetDomainFactories.ImageFactory.Get(mediaItem.MediaAssetID);
				DomainFactories.AssetDomainFactories.AssetVersionFactory.Create(image, Roblox.Platform.Assets.CreatorType.User, authenticatedUser.Id, null, resampledTexture, streamCreatorInfo.DecompressionMethod);
			}
			else
			{
				bool isPlace = asset.AssetTypeID == AssetType.PlaceID;
				UploadAssetMediaImage(resampledTexture, asset, authenticatedUser, isPlace);
			}
		}
		return duration;
	}

	private void UploadGeneratedAssetMediaImageAndUploadCounters(Asset asset, IUser authenticatedIUser)
	{
		IncrementEphemeralCounter("AddGeneratedModel_Attempt");
		try
		{
			RecordAddGeneratedModelProcessingTimeInEphemeralCounter(DoUpload(asset, authenticatedIUser).TotalMilliseconds);
		}
		catch (Exception ex)
		{
			IncrementEphemeralCounter("AddGeneratedModel_Error");
			ExceptionHandler.LogException(ex);
		}
	}

	private void RecordAddGeneratedModelProcessingTimeInEphemeralCounter(double timeInMilliSeconds)
	{
		ISequence counter = DomainFactories.EphemeralCounterFactory.GetSequence("AddGeneratedModel_ProccesingTime");
		RobloxThreadPool.QueueUserWorkItem(delegate
		{
			counter.Add(timeInMilliSeconds);
		});
	}

	private void IncrementEphemeralCounter(string counterName)
	{
		ICounter counter = DomainFactories.EphemeralCounterFactory.GetCounter(counterName);
		RobloxThreadPool.QueueUserWorkItem(delegate
		{
			counter.Increment(1);
		});
	}
}
